<?php

namespace RageKings\KingWalletApi\Http\Middleware;

use GuzzleHttp\Exception\RequestException;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Sentry\SentrySdk;
use Sentry\State\HubInterface;
use Sentry\Tracing\SpanContext;
use Closure;
use Throwable;

class HttpTracingMiddleware
{
    public static function trace(?HubInterface $hub = null): Closure
    {
        return function (callable $handler) use ($hub): Closure {
            return function (RequestInterface $request, array $options) use ($hub, $handler) {
                $hub = $hub ?? SentrySdk::getCurrentHub();
                $span = $hub->getSpan();
                $childSpan = null;

                if (null !== $span) {
                    $spanContext = new SpanContext();
                    $spanContext->setOp('http.guzzle');
                    $spanContext->setDescription($request->getMethod() . ' ' . $request->getUri());

                    $spanContext->setData([
                        'request-headers' => $request->getHeaders(),
                        'request-body' => (string) $request->getBody()
                    ]);

                    $childSpan = $span->startChild($spanContext);
                }

                $handlerPromiseCallback = static function ($responseOrException) use ($childSpan) {
                    if (null !== $childSpan) {
                        $childSpan->finish();
                        if ($responseOrException instanceof RequestException) {
                            if ($responseOrException->getResponse() !== null) {
                                $response = $responseOrException->getResponse();

                                $childSpan->setData([
                                    'response-headers' => $response->getHeaders(),
                                    'response-body' => (string) $response->getBody(),
                                    'response-status' => $response->getStatusCode()
                                ]);
                            }
                        }
                    }

                    if ($responseOrException instanceof Throwable) {
                        throw $responseOrException;
                    }

                    if (null !== $childSpan) {
                        /** @var ResponseInterface $responseOrException */
                        $childSpan->setData([
                            'response-headers' => $responseOrException->getHeaders(),
                            'response-body' => (string) $responseOrException->getBody(),
                            'response-status' => $responseOrException->getStatusCode()
                        ]);
                    }

                    return $responseOrException;
                };

                return $handler($request, $options)->then($handlerPromiseCallback, $handlerPromiseCallback);
            };
        };
    }
}