<?php

namespace RageKings\KingPayApi\Responses\Invoice;

use Illuminate\Support\Carbon;
use RageKings\KingPayApi\Casters\CarbonCaster;
use RageKings\KingPayApi\Casters\Invoice\StateCaster;
use RageKings\KingPayApi\Enums\Invoice\State;
use Spatie\DataTransferObject\Attributes\CastWith;
use Spatie\DataTransferObject\DataTransferObject;

class Invoice extends DataTransferObject
{
    /**
     * Уникальный ID счета в нашей системе. Формируется в формате UUID
     * @var string
     */
    public string $id;

    /**
     * Уникальный ID счета для продавца. Это может быть номер заказа в системе продавца
     * @var string
     */
    public string $customer_invoice_id;

    /**
     * Уникальный ID проекта
     * @var string|null
     */
    public ?string $project_id;

    /**
     * Заголовок счет-фактуры
     * @var string
     */
    public string $title;

    /**
     * Описание счет-фактуры
     * @var string
     */
    public string $comment;

    /**
     * Информация по сумме фактуры
     * @var Sum
     */
    public Sum $sum;

    /**
     * Сумма, на которую была закрыта счет-фактура. Колонка выводится только когда счёт-фактура имеет состояниен success
     * @var float|null
     */
    public ?float $final_amount;

    /**
     * Состояние счет-фактуры (примеры: new, payment_wait, payed, success, canceled, autocanceled)
     * @var State
     */
    #[CastWith(StateCaster::class)]
    public State $state;

    /**
     * Ссылка на форму оплаты счет-фактуры
     * @var string
     */
    public string $invoice_url;

    /**
     * URL на который произойдет редирект пользователя после удачной оплаты счет-фактуры
     * @var string|null
     */
    public ?string $successUrl;

    /**
     * URL на который произойдет редирект пользователя, если счет-фактура была отменена или истек ее срок действия
     * @var string|null
     */
    public ?string $failUrl;

    /**
     * Дата действия счет-фактуры в формате ATOM
     * @var Carbon
     */
    #[CastWith(CarbonCaster::class)]
    public Carbon $expired_at;

    /**
     * Дата создания счет-фактуры в формате ATOM
     * @var Carbon
     */
    #[CastWith(CarbonCaster::class)]
    public Carbon $created_at;

    /**
     * Текущая активная заявка на оплату. Значение возвращается только при наличии заявки
     * @var InvoiceOrder|null
     */
    public ?InvoiceOrder $current_order;

    /**
     * Указанные теги к счёт-фактуре. Значение возвращается только при наличии тегов
     * @var array|null
     */
    public ?array $tags;

    /**
     * Получение суммы счёт-фактуры
     * @return float
     */
    public function getAmount(): float
    {
        return $this->sum->money->amount;
    }
}