<?php

namespace Astreya\WalletUtils\Dto;

use InvalidArgumentException;

class WalletStates
{
    const outgoing_limits_state             = 'outgoing_limits';                       // Запрет на исходящие платежи. В данное время нельзя провести исходящий перевод
    const incoming_limits_state             = 'incoming_limits';                       // Запрет на входящие платежи. В данное время нельзя провести входящий перевод
    const permanently_blocked_state         = 'permanently_blocked';                   // Полная блокировка. Это окончательная блокировка, которая означает полную смерть кошелька
    const partial_blocked_state             = 'partial_blocked';                       // Частичная блокировка. Частичная блокировка кошелька. Доступ к нему есть, но его функционал ограничен
    const temporary_blocked_state           = 'temporary_blocked';                     // Временная блокировка. Через какое-то время она сама пропадет
    const need_real_person_verification_state  = 'need_real_person_verification';      // Нужна проверка личности в реальной жизни
    const need_digital_person_verification_state = 'need_digital_person_verification'; // Нужна проверка личности в интернете. Ввод паспортных данных, фото и т.д.

    const SUPPORTED_STATES = [
        self::incoming_limits_state,
        self::outgoing_limits_state,
        self::permanently_blocked_state,
        self::partial_blocked_state,
        self::temporary_blocked_state,
        self::need_real_person_verification_state,
        self::need_digital_person_verification_state
    ];

    protected array $states = [];

    public function __construct()
    {
        foreach (self::SUPPORTED_STATES as $state) {
            $this->states[$state] = null;
        }
    }

    /**
     * @return array
     */
    public function getStates(): array
    {
        return $this->states;
    }

    /**
     * @param array $states
     */
    public function setStates(array $states): void
    {
        $this->states = $states;
    }

    /**
     * Установка состояния для кошелька
     *
     * @param string $state
     * @param bool $value
     * @return void
     */
    public function setState(string $state, bool $value = null): void
    {
        if (in_array($state, self::SUPPORTED_STATES) !== true)
            throw new InvalidArgumentException("Invalid wallet state $state");

        $this->states[$state] = $value;
    }

    /**
     * Получение состояния кошелька
     *
     * @param string $state
     * @return bool|null
     */
    public function getState(string $state): ?bool
    {
        if (in_array($state, self::SUPPORTED_STATES) !== true)
            throw new InvalidArgumentException("Invalid wallet state $state");

        return $this->states[$state];
    }

    public static function createFromArray(array $data): self
    {
        $ob = new self();
        foreach($data as $key => $value) {
            $ob->setState($key, $value);
        }
        return $ob;
    }

    public function toArray(): array
    {
        return $this->getStates();
    }
}