<?php


namespace Astreya\WalletUtils\Dto;


use Carbon\Carbon;
use InvalidArgumentException;

class WalletMessage
{
    const TYPE_SMS      = 'sms';
    const TYPE_USSD     = 'ussd';
    const TYPE_MAIL     = 'mail';
    const TYPE_CUSTOM   = 'custom';

    const DIRECTION_INCOMING = 'incoming';
    const DIRECTION_OUTGOING = 'outgoing';

    const SUPPORTED_TYPES = [
      self::TYPE_SMS,
      self::TYPE_MAIL,
      self::TYPE_USSD,
      self::TYPE_CUSTOM
    ];

    const SUPPORTED_DIRECTIONS = [
      self::DIRECTION_INCOMING, self::DIRECTION_OUTGOING
    ];

    private string $to;

    private string $from;

    private string $type;

    private string $direction;

    private string $subject;

    private string $body;

    private Carbon $created_at;

    private array $extra = [];

    /**
     * @return string
     */
    public function getTo(): string
    {
        return $this->to;
    }

    /**
     * @param string $to
     */
    public function setTo(string $to): void
    {
        $this->to = $to;
    }

    /**
     * @return string
     */
    public function getFrom(): string
    {
        return $this->from;
    }

    /**
     * @param string $from
     */
    public function setFrom(string $from): void
    {
        $this->from = $from;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @param string $type
     */
    public function setType(string $type): void
    {
        if (in_array($type, self::SUPPORTED_TYPES) === false)
            throw new InvalidArgumentException("unsupported event $type");

        $this->type = $type;
    }



    /**
     * @return string
     */
    public function getDirection(): string
    {
        return $this->direction;
    }

    /**
     * @param string $direction
     */
    public function setDirection(string $direction): void
    {
        if (in_array($direction, self::SUPPORTED_DIRECTIONS) === false)
            throw new InvalidArgumentException("unsupported direction $direction");

        $this->direction = $direction;
    }

    /**
     * @return string
     */
    public function getSubject(): string
    {
        return $this->subject;
    }

    /**
     * @param string $subject
     */
    public function setSubject(string $subject): void
    {
        $this->subject = $subject;
    }

    /**
     * @return string
     */
    public function getBody(): string
    {
        return $this->body;
    }

    /**
     * @param string $body
     */
    public function setBody(string $body): void
    {
        $this->body = $body;
    }



    /**
     * @return \Carbon\Carbon
     */
    public function getCreatedAt(): Carbon
    {
        return $this->created_at;
    }

    /**
     * @param \Carbon\Carbon|string $created_at
     */
    public function setCreatedAt($created_at): void
    {
        if (is_string($created_at))
            $created_at = Carbon::parse($created_at);

        $this->created_at = $created_at;
    }

    /**
     * @return array
     */
    public function getExtra(): array
    {
        return $this->extra;
    }

    /**
     * @param array $extra
     */
    public function setExtra(array $extra): void
    {
        $this->extra = $extra;
    }

    /**
     * @param string $json
     * @return static
     */
    public static function createFromJson(string $json): self
    {
        return self::createFromArray(json_decode($json, true));
    }

    /**
     * @param array $data
     * @return static
     */
    public static function createFromArray(array $data): self
    {
        $ob = new self();
        $ob->setTo($data['to']);
        $ob->setFrom($data['from']);
        $ob->setType($data['type']);
        $ob->setDirection($data['direction']);
        $ob->setSubject($data['subject']);
        $ob->setBody($data['body']);
        $ob->setCreatedAt($data['created_at']);
        if (array_key_exists('extra', $data))
            $ob->setExtra($data['extra']);

        return $ob;
    }

    /**
     * @return array
     */
    public function toArray(): array
    {
        return [
            'to'            => $this->getTo(),
            'from'          => $this->getFrom(),
            'type'          => $this->getType(),
            'direction'     => $this->getDirection(),
            'subject'       => $this->getSubject(),
            'body'          => $this->getBody(),
            'created_at'    => $this->getCreatedAt()->format('Y-m-d\TH:i:sP'),
            'extra'         => $this->getExtra()
        ];
    }

}