<?php


namespace Astreya\WalletUtils\Dto\Response;


use Carbon\Carbon;
use InvalidArgumentException;
use Psr\Http\Message\ResponseInterface;

class FetchedBalance
{
    const SUCCESS = 'success';
    const FAILED = 'failed';

    const RESULTS = [self::SUCCESS, self::FAILED];

    private string $result;

    private string $message = '';

    private ?float $balance;

    private ?Carbon $balance_checked_at;

    /**
     * @return string
     */
    public function getResult(): string
    {
        return $this->result;
    }

    /**
     * @param string $result
     * @throws \InvalidArgumentException
     */
    private function setResult(string $result): void
    {
        if (true !== in_array($result, self::RESULTS))
            throw new InvalidArgumentException("$result does not valid result");

        $this->result = $result;
    }

    /**
     * @return string
     */
    public function getMessage(): string
    {
        return $this->message;
    }

    /**
     * @param string $message
     */
    public function setMessage(string $message): void
    {
        $this->message = $message;
    }



    /**
     * @return float|null
     */
    public function getBalance(): ?float
    {
        return $this->balance;
    }

    /**
     * @param float|null $balance
     */
    private function setBalance(?float $balance): void
    {
        $this->balance = $balance;
    }

    /**
     * @return \Carbon\Carbon|null
     */
    public function getBalanceCheckedAt(): ?Carbon
    {
        return $this->balance_checked_at;
    }

    /**
     * @param \Carbon\Carbon|string|null $balance_checked_at
     */
    private function setBalanceCheckedAt($balance_checked_at): void
    {
        if (is_string($balance_checked_at))
            $balance_checked_at = Carbon::parse($balance_checked_at);

        $this->balance_checked_at = $balance_checked_at;
    }

    /**
     * Создание объекта из ответа на Http-запрос
     *
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return static
     */
    static function createFromHttpResponse(ResponseInterface $response) : self
    {
        $json = (string)$response->getBody();
        $data = json_decode($json, true);
        return self::createFromArray($data);
    }

    /**
     * Создание объекта из ответа на Http-запрос
     *
     * @param string $json
     * @return static
     */
    static function createFromJson(string $json) : self
    {
        $data = json_decode($json, true);
        return self::createFromArray($data);
    }

    /**
     * Создание обьекта из массива
     *
     * @param array $data
     * @return static
     */
    static function createFromArray(array $data) : self
    {
        $ob = new self();
        $ob->setResult($data['result']);

        if (isset($data['message']))
            $ob->setMessage($data['message']);

        $ob->setBalance($data['data']['balance']);
        $ob->setBalanceCheckedAt($data['data']['balance_checked_at']);

        return $ob;
    }

    public function toArray() : array
    {
        return [
            'result'    => $this->getResult(),
            'message'   => $this->getMessage(),
            'data'      => [
                'balance_checked_at'    => $this->getBalanceCheckedAt()->format('Y-m-d\TH:i:sP'),
                'balance'               => $this->getBalance()
            ]
        ];
    }
}
