<?php


namespace Astreya\WalletUtils\Dto;


use InvalidArgumentException;

class WalletEvent
{
    const BLOCK_WALLET_EVENT                = 'block_wallet';
    const WALLET_WITH_OUTGOING_LIMITS_EVENT = 'wallet_with_outgoing_limits';
    const WALLET_WITH_INCOMING_LIMITS_EVENT = 'wallet_with_incoming_limits';
    const WALLET_LOGIN_SUCCESSFUL           = 'wallet_login_successful';
    const WALLET_SIGNUP_SUCCESSFUL          = 'wallet_signup_successful';
    const WALLET_PREPARATION_SUCCESSFUL     = 'wallet_preparation_successful';
    const FETCHED_BALANCE_EVENT             = 'fetched_balance';
    const FETCHED_PAYMENTS_EVENT            = 'fetched_payments';
    const FETCHED_PROFILE_EVENT             = 'fetched_profile';
    const UNAUTHORIZED_WALLET_EVENT         = 'unauthorized_wallet';

    const SUPPORTED_EVENTS = [
        self::WALLET_WITH_INCOMING_LIMITS_EVENT,
        self::WALLET_WITH_OUTGOING_LIMITS_EVENT,
        self::BLOCK_WALLET_EVENT,
        self::WALLET_LOGIN_SUCCESSFUL,
        self::WALLET_PREPARATION_SUCCESSFUL,
        self::WALLET_SIGNUP_SUCCESSFUL,
        self::FETCHED_BALANCE_EVENT,
        self::FETCHED_PAYMENTS_EVENT,
        self::FETCHED_PROFILE_EVENT,
        self::UNAUTHORIZED_WALLET_EVENT
    ];

    private string $event;

    private WalletData $walletData;

    private array $context = [];

    /**
     * @return string
     */
    public function getEvent(): string
    {
        return $this->event;
    }

    /**
     * @param string $event
     */
    public function setEvent(string $event): void
    {
        if (in_array($event, self::SUPPORTED_EVENTS) === false)
            throw new InvalidArgumentException("unsupported event $event");

        $this->event = $event;
    }

    /**
     * @return \Astreya\WalletUtils\Dto\WalletData
     */
    public function getWalletData(): WalletData
    {
        return $this->walletData;
    }

    /**
     * @param \Astreya\WalletUtils\Dto\WalletData|array $walletData
     */
    public function setWalletData($walletData): void
    {
        if (is_array($walletData))
            $walletData = WalletData::createFromArray($walletData);

        $this->walletData = $walletData;
    }

    /**
     * @return array
     */
    public function getContext(): array
    {
        return $this->context;
    }

    /**
     * @param array $context
     */
    public function setContext(array $context): void
    {
        $this->context = $context;
    }

    public function toArray(): array
    {
        return [
            'event'     => $this->getEvent(),
            'wallet'    => $this->getWalletData()->toArray(),
            'context'   => $this->getContext()
        ];
    }

    static function createFromArray(array $data): self
    {
        $ob = new self();
        $ob->setEvent($data['event']);
        $ob->setWalletData($data['wallet']);
        if (isset($data['context'])) {
            $ob->setContext($data['context']);
        }
        return $ob;
    }

    static function createFromJson(string $data): self
    {
        return self::createFromArray(json_decode($data, true));
    }
}