<?php


namespace Astreya\WalletUtils\Dto;


use Carbon\Carbon;
use InvalidArgumentException;

class PaymentData
{
    CONST STATE_NEW         = 'new';
    CONST STATE_SUCCESS     = 'success';
    CONST STATE_ERROR       = 'error';
    CONST STATE_ACCEPTED    = 'accepted';       // Для исходящих платежей, которые прошли все этапы формирования платежа, но сторонний сервис пока не подтвердил платеж
    CONST STATE_CANCELED    = 'canceled';       // Для отмененных или просроченных платежей
    CONST STATE_IN_PROGRESS = 'in_progress';    // Для исходящих платежей, находящихся в состоянии выполнения
    CONST STATE_REFUNDED    = 'refunded';       // Для систем, которые успешный платеж могут перевести в рефанд. Использовать только при крайней необходимости
    CONST STATE_UNCONFIRMED = 'unconfirmed';    // Для криптовалюты, когда платеж проведен, но не имеет достаточных подтверждений

    const DIRECTION_INCOMING = 0;
    const DIRECTION_OUTGOING = 1;

    const STATES = [
        self::STATE_NEW,
        self::STATE_SUCCESS,
        self::STATE_CANCELED,
        self::STATE_ERROR,
        self::STATE_ACCEPTED,
        self::STATE_IN_PROGRESS,
        self::STATE_UNCONFIRMED,
        self::STATE_REFUNDED
    ];

    const DIRECTIONS = [
        self::DIRECTION_INCOMING,
        self::DIRECTION_OUTGOING
    ];

    private string $transaction_id = '';

    /**
     * ID транзакции-родителя. Необходимо указывать в случаях, когда в рамках одной транзакции осуществляется
     * несколько платежей. Например, при массовых выплатах криптовалютой
     *
     * @var string
     */
    private string $parent_transaction_id = '';

    private string $title = '';

    /**
     * Размер суммы перевода
     *
     * @var float
     */
    private float $sum_amount;

    private string $sum_currency;

    /**
     * Размер комиссии за перевод
     *
     * @var float
     */
    private float $fee_amount = 0;

    private string $state = self::STATE_NEW;

    private int $direction;

    private Carbon $created_at;

    private ?string $dst_model_type = null;

    private ?string $dst_model_id = null;

    private ?string $instance = null;

    private array $extra = [];

    /**
     * @return string
     */
    public function getState(): string
    {
        return $this->state;
    }

    /**
     * @param string $state
     */
    public function setState(string $state): void
    {
        if (true !== in_array($state, self::STATES))
            throw new InvalidArgumentException("$state does not valid state");

        $this->state = $state;
    }

    /**
     * @return int
     */
    public function getDirection(): int
    {
        return $this->direction;
    }

    /**
     * @param int $direction
     */
    public function setDirection(int $direction): void
    {
        if (true !== in_array($direction, self::DIRECTIONS))
            throw new InvalidArgumentException("$direction - does not valid direction");

        $this->direction = $direction;
    }

    /**
     * @return float
     */
    public function getSumAmount(): float
    {
        return $this->sum_amount;
    }

    /**
     * @param float $sum_amount
     */
    public function setSumAmount(float $sum_amount): void
    {
        $this->sum_amount = $sum_amount;
    }

    /**
     * @return string
     */
    public function getSumCurrency(): string
    {
        return $this->sum_currency;
    }

    /**
     * @param string $sum_currency
     */
    public function setSumCurrency(string $sum_currency): void
    {
        $this->sum_currency = $sum_currency;
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @param string $title
     */
    public function setTitle(string $title): void
    {
        $this->title = $title;
    }

    /**
     * @return string
     */
    public function getTransactionId(): string
    {
        return $this->transaction_id;
    }

    /**
     * @param string $transaction_id
     */
    public function setTransactionId(string $transaction_id): void
    {
        $this->transaction_id = $transaction_id;
    }

    /**
     * @return string
     */
    public function getCreatedAt(): string
    {
        return $this->created_at->format('Y-m-d\TH:i:sP');
    }

    /**
     * @param string|Carbon $created_at
     */
    public function setCreatedAt($created_at): void
    {
        if (is_string($created_at))
            $created_at = Carbon::parse($created_at);

        $this->created_at = $created_at;
    }

    /**
     * @return string|null
     */
    public function getDstModelType(): ?string
    {
        return $this->dst_model_type;
    }

    /**
     * @param string|null $dst_model_type
     */
    public function setDstModelType(?string $dst_model_type): void
    {
        $this->dst_model_type = $dst_model_type;
    }

    /**
     * @return string|null
     */
    public function getDstModelId(): ?string
    {
        return $this->dst_model_id;
    }

    /**
     * @param string|null $dst_model_id
     */
    public function setDstModelId(?string $dst_model_id): void
    {
        $this->dst_model_id = $dst_model_id;
    }

    /**
     * @return array
     */
    public function getExtra(): array
    {
        return $this->extra;
    }

    /**
     * @param array $extra
     */
    public function setExtra(array $extra): void
    {
        $this->extra = $extra;
    }

    /**
     * @return string|null
     */
    public function getInstance(): ?string
    {
        return $this->instance;
    }

    /**
     * @param string|null $instance
     */
    public function setInstance(?string $instance): void
    {
        $this->instance = $instance;
    }

    /**
     * Получение комиссии за перевод
     *
     * @return float
     */
    public function getFeeAmount(): float
    {
        return $this->fee_amount;
    }

    /**
     * Установка комиссии за перевод
     *
     * @param float $fee_amount
     */
    public function setFeeAmount(float $fee_amount): void
    {
        $this->fee_amount = $fee_amount;
    }

    /**
     * Получение ID транзакции-родителя
     *
     * @return string
     */
    public function getParentTransactionId(): string
    {
        return $this->parent_transaction_id;
    }

    /**
     * Установка ID транзакции-родителя
     *
     * @param string $parent_transaction_id
     */
    public function setParentTransactionId(string $parent_transaction_id): void
    {
        $this->parent_transaction_id = $parent_transaction_id;
    }

    /**
     * Создание обьекта из массива
     *
     * @param array $data
     * @return static
     */
    static function createFromArray(array $data) : self
    {
        $ob = new self();

        if (isset($data['state'])) {
            $ob->setState($data['state']);
        }

        $ob->setDirection($data['direction']);

        if (isset($data['transaction_id'])) {
            $ob->setTransactionId($data['transaction_id']);
        }

        if (isset($data['parent_transaction_id'])) {
            $ob->setParentTransactionId($data['parent_transaction_id']);
        }

        $ob->setSumAmount($data['sum_amount']);
        $ob->setSumCurrency($data['sum_currency']);

        if (isset($data['fee_amount']))
            $ob->setFeeAmount($data['fee_amount']);

        if (isset($data['title'])) {
            $ob->setTitle($data['title']);
        }

        $ob->setCreatedAt($data['created_at'] ?? now());

        if (isset($data['dst_model_type']) && isset($data['dst_model_id'])) {
            $ob->setDstModelType($data['dst_model_type']);
            $ob->setDstModelId($data['dst_model_id']);
        }

        if (isset($data['extra'])) {
            $ob->setExtra($data['extra']);
        }

        if (isset($data['instance'])) {
            $ob->setInstance($data['instance']);
        }

        return $ob;
    }

    public function toArray(): array
    {
        $data = [
            'state'             => $this->getState(),
            'direction'         => $this->getDirection(),
            'transaction_id'    => $this->getTransactionId(),
            'sum_amount'        => $this->getSumAmount(),
            'sum_currency'      => $this->getSumCurrency(),
            'fee_amount'        => $this->getFeeAmount(),
            'title'             => $this->getTitle(),
            'created_at'        => $this->getCreatedAt(),
            'instance'          => $this->getInstance(),
        ];

        if ($this->getParentTransactionId())
            $data['parent_transaction_id'] = $this->getParentTransactionId();

        if ($this->getDstModelId() && $this->getDstModelType()) {
            $data['dst_model_type'] = $this->getDstModelType();
            $data['dst_model_id'] = $this->getDstModelId();
        }

        $data['extra'] = $this->getExtra();

        return $data;
    }

}
