<?php

namespace Astreya\TaskSystem\Tests;

use Astreya\TaskSystem\Providers\TaskServiceProvider;
use Astreya\WalletUtils\Dto\PaymentData;
use Astreya\WalletUtils\Dto\WalletData;
use Astreya\WalletUtils\Dto\WalletMessage;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Carbon;
use Orchestra\Testbench\TestCase;
use Sentry\Laravel\ServiceProvider;
use Sentry\SentrySdk;
use Sentry\Tracing\Transaction;
use Sentry\Tracing\TransactionContext;

abstract class IntegrationTest extends TestCase
{
    use RefreshDatabase, WithFaker;
    /**
     * @throws \Exception
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->loadMigrations();
        $this->withFactories(__DIR__.'/Factories');
    }

    /**
     * Load the migrations for the test environment.
     *
     * @return void
     */
    protected function loadMigrations(): void
    {
        $this->loadMigrationsFrom([
            '--path' => realpath(__DIR__.'/Migrations'),
        ]);
    }

    protected function getPackageProviders($app): array
    {
        return [
            TaskServiceProvider::class,
            ServiceProvider::class      // Подключаем сентри, что бы иметь доступ к app()->get('sentry')
        ];
    }

    protected function buildWalletData(): WalletData
    {
        $walletData = new WalletData();
        $walletData->setLogin($this->faker->phoneNumber);
        $walletData->setPhoneNumber($this->faker->phoneNumber);
        $walletData->setPassword($this->faker->password);
        $walletData->setEmail($this->faker->email);

        return $walletData;
    }

    protected function buildMessageData(): WalletMessage
    {
        $messageData = new WalletMessage();
        $messageData->setTo($this->faker->phoneNumber);
        $messageData->setSubject($this->faker->title);
        $messageData->setCreatedAt($this->faker->date);
        $messageData->setFrom($this->faker->phoneNumber);
        $messageData->setBody($this->faker->text());
        $messageData->setType($this->faker->randomElement(WalletMessage::SUPPORTED_TYPES));
        $messageData->setDirection($this->faker->randomElement(WalletMessage::SUPPORTED_DIRECTIONS));

        return $messageData;
    }

    protected function buildPaymentData(): PaymentData
    {
        $paymentData = new PaymentData();
        $paymentData->setDirection($this->faker->randomElement(PaymentData::DIRECTIONS));
        $paymentData->setState($this->faker->randomElement(PaymentData::STATES));
        $paymentData->setSumAmount($this->faker->numberBetween(1, 15000));
        $paymentData->setSumCurrency('RUB');
        $paymentData->setCreatedAt(Carbon::parse($this->faker->dateTimeThisMonth));

        return $paymentData;
    }

    /**
     * Генерация транзакции Sentry
     *
     * @param string $name
     * @return \Sentry\Tracing\Transaction
     */
    protected function startSentryTransaction(string $name): Transaction
    {
        $tc = new TransactionContext();
        $tc->setName($name);
        $tc->setOp('test');

        $transaction = SentrySdk::getCurrentHub()->startTransaction($tc);
        SentrySdk::getCurrentHub()->setSpan($transaction);

        return $transaction;
    }
}