<?php

namespace Astreya\TaskSystem\Http\Controllers\Api\V1;

use Astreya\TaskSystem\Http\Requests\CreateTaskRequest;
use Astreya\TaskSystem\Jobs\ProcessTaskJob;
use Astreya\TaskSystem\Models\QueuedTask;
use Astreya\TaskSystem\Services\TaskService;
use Astreya\WalletUtils\Dto\QueuedTaskData;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;

class TaskController extends Controller
{
    const RESULT_SUCCESS    = 'success';
    const RESULT_ERROR      = 'error';

    /**
     * Просмотр информации по задаче
     *
     * @param \Astreya\TaskSystem\Models\QueuedTask $task
     * @return \Illuminate\Http\JsonResponse
     */
    public function view(QueuedTask $task): JsonResponse
    {
        return response()->json([
            'result'    => self::RESULT_SUCCESS,
            'data'      => $task->toTaskData()->toArray()
        ]);
    }

    /**
     * Обработка задачи
     *
     * @param \Astreya\TaskSystem\Models\QueuedTask $task
     * @return \Illuminate\Http\JsonResponse
     */
    public function process(QueuedTask $task) : JsonResponse
    {
        dispatch_sync(new ProcessTaskJob($task));
        $task->refresh(); //TODO проблема в том что dispatch_sync() не меняет переменную как dispatch_now()

        return response()->json([
            'result' => self::RESULT_SUCCESS,
            'data' => $task->toTaskData()->toArray()
        ]);
    }

    /**
     * Отмена задачи
     *
     * @param \Astreya\TaskSystem\Models\QueuedTask $task
     * @return \Illuminate\Http\JsonResponse
     */
    public function reject(QueuedTask $task) : JsonResponse
    {
        $task->update([
            'state' => QueuedTaskData::STATE_REJECTED
        ]);

        return response()->json([
            'result' => self::RESULT_SUCCESS,
            'data' => $task->toTaskData()->toArray()
        ]);
    }

    /**
     * Создание задачи
     *
     * @param \Astreya\TaskSystem\Http\Requests\CreateTaskRequest $request
     * @param \Astreya\TaskSystem\Services\TaskService $taskService
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(CreateTaskRequest $request, TaskService $taskService) : JsonResponse
    {
        $task = $taskService->initQueuedTask($request->getTaskType(), $request->getTaskContext(), $request->getTaskUniqueId());

        return response()->json([
            'result' => self::RESULT_SUCCESS,
            'data' => $task->toTaskData()->toArray()
        ]);
    }
}