<?php


namespace Astreya\TaskSystem\Processors;

use Astreya\TaskSystem\Models\QueuedTask;
use Astreya\TaskSystem\Contracts\TaskProcessor;
use Astreya\TaskSystem\Traits\WithSentry;
use Astreya\WalletUtils\Dto\QueuedTaskData;
use Astreya\WalletUtils\Dto\WalletData;
use Sentry\State\Scope;
use Sentry\Tracing\SpanStatus;
use Throwable;
use function Sentry\captureException;
use function Sentry\withScope;

abstract class BaseTaskProcessor implements TaskProcessor
{
    use WithSentry;

    protected const DEFAULT_STEP = 'begin_step';
    protected const PROCESS_CONTEXT_KEY = '_process';

    protected QueuedTask $currentTask;

    /**
     * Информация о кошельке, для которого выполняется задача
     */
    protected WalletData $walletData;

    protected array $taskContext = [];

    public function __construct(QueuedTask $task)
    {
        $this->currentTask = $task;
    }

    /**
     * Установка задачи в процессор
     *
     * @param \Astreya\TaskSystem\Models\QueuedTask $task
     * @return void
     */
    public function setTask(QueuedTask $task): void
    {
        $this->currentTask = $task;
    }

    /**
     * @return void
     * @uses setStep
     * @uses begin_step
     */
    public function run(): void
    {
        $this->setTaskContext($this->currentTask->context);
        $this->setProcessorContext($this->currentTask->context[self::PROCESS_CONTEXT_KEY] ?? []);
        $step = $this->getStep();

        withScope(function(Scope $scope){
            if ($scope->getTransaction()) {
                $scope->getTransaction()->setTags(['step' => $this->getStep()]);
            }
        });

        try {
            $this->$step($this->getStep());
            if ($this->getSentry()->getTransaction()) {
                $this->getSentry()->getTransaction()->setStatus(SpanStatus::ok());
            }
        } catch (Throwable $exception) {
            if ($this->getSentry()->getTransaction()) {
                $this->getSentry()->getTransaction()->setStatus(SpanStatus::internalError());
            }
            $this->failed($exception);
        }

        $this->currentTask->context = $this->getTaskContext(); /* TODO: Можно заменить на временный контекст внутри самой модели */
    }

    protected function getStep(): string
    {
        $pc = $this->getProcessorContext();
        return $pc['step'] ?? self::DEFAULT_STEP;
    }

    /**
     * Установка текущего этапа для задачи
     * @param string $step
     */
    protected function setStep(string $step): void
    {
        $this->taskContext[self::PROCESS_CONTEXT_KEY]['step'] = $step;
    }

    private function getProcessorContext(): array
    {
        return $this->taskContext[self::PROCESS_CONTEXT_KEY];
    }

    private function setProcessorContext(array $context): void
    {
        $this->taskContext[self::PROCESS_CONTEXT_KEY] = $context;
    }

    private function setTaskContext(array $context): void
    {
        $this->taskContext = $context;
    }

    private function getTaskContext(): array
    {
        return $this->taskContext;
    }

    abstract protected function begin_step(): void;

    protected function failed(Throwable $exception): void
    {
        captureException($exception);
        $this->currentTask->result = [
            'error' => [
                'code'    => $exception->getCode(),
                'message' => $exception->getMessage()
            ]
        ];

        $this->currentTask->state = QueuedTaskData::STATE_FAILED;
    }
}
