<?php

namespace Astreya\TaskSystem\Jobs;

use Astreya\TaskSystem\Contracts\TaskProcessor;
use Astreya\TaskSystem\Jobs\Middleware\WithSentryTransaction;
use Astreya\TaskSystem\Models\QueuedTask;
use Astreya\WalletUtils\Dto\QueuedTaskData;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUniqueUntilProcessing;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Throwable;


class ProcessTaskJob implements ShouldQueue, ShouldBeUniqueUntilProcessing
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var QueuedTask
     */
    private QueuedTask $task;

    protected TaskProcessor $processor;

    /**
     * Количество секунд, по истечении которых уникальная блокировка задания будет снята.
     *
     * @var int
     */
    public int $uniqueFor = 3600;

    /**
     * Уникальный идентификатор задания.
     *
     * @return string
     */
    public function uniqueId(): string
    {
        return $this->task->id;
    }

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(QueuedTask $task)
    {
        //
        $this->task = $task;
    }

    /**
     * Execute the job.
     *
     * @return void
     * @throws \Throwable
     */
    public function handle(): void
    {
        //
        try {
            Log::info("Обработка задачи {$this->task->id} типа {$this->task->type}");
            /** @var string $processor_class */
            $processor_class = config('astreya-tasks.task_processors.' . $this->task->type);
            $this->processor = new $processor_class($this->task);
            $this->processor->run();
            $this->task->save();
        } catch (Throwable $exception) {
            $this->task->state = QueuedTaskData::STATE_FAILED;
            $this->task->save();
            Log::error("Ошибка при обработке задачи {$this->task->id} типа {$this->task->type}");
            throw $exception;
        }
    }

    /**
     * Get the middleware the job should pass through.
     *
     * @return array
     */
    public function middleware(): array
    {
        return [new WithSentryTransaction('process_task')];
    }
}
