<?php

namespace Astreya\TaskSystem\Console\Commands;

use Astreya\TaskSystem\Jobs\ProcessTaskJob;
use Astreya\TaskSystem\Models\QueuedTask;
use Astreya\TaskSystem\Services\TaskService;
use Astreya\WalletUtils\Dto\QueuedTaskData;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;
use Throwable;

class ProcessTasksCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'task:run {type} {--sync}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Запуск обработки задачи';

    protected string $type;

    public function handle(TaskService $taskService): int
    {
        $this->type = (string)$this->argument('type');
        $this->info("Запуск процесса обработки задач типа $this->type");

        if (in_array($this->type, QueuedTaskData::TYPES) !== true)
            throw new InvalidArgumentException("$this->type - некорректный тип задачи");

        $tasks = $taskService->getTaskForProcess($this->type);

        if ($tasks->count() === 0)
            $this->info("Отсутствуют задачи типа $this->type для обработки");

        $tasks->each(function(QueuedTask $task) {
            $this->info("Обработка задачи $task->id типа $task->type");
            $job = new ProcessTaskJob($task);
            $job->onQueue("{$this->type}_tasks");
            try {
                if ($this->option('sync'))
                    dispatch_sync($job);
                else
                    dispatch($job);
            } catch (Throwable $exception) {
                Log::error("Ошибка при обработке задачи типа $task->type c UID: $task->id");
            }
        });

        return self::SUCCESS;
    }
}