<?php

namespace Astreya\TaskSystem\Services;

use Astreya\TaskSystem\Models\QueuedTask;
use Astreya\WalletUtils\Dto\QueuedTaskData;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class TaskService
{
    /**
     * @param string $type
     * @param array $context
     * @param string|null $unique_id
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initQueuedTask(string $type, array $context, string $unique_id = null) : QueuedTask
    {
        /* Защита от дублей */
        if ($unique_id) {
            $task = QueuedTask::where('unique_id', '=', $unique_id)
                ->first();

            if ($task)
                return $task;
        }

        return QueuedTask::create([
            'state'     => QueuedTaskData::STATE_IN_QUEUE,
            'type'      => $type,
            'id'        => Str::uuid()->toString(),
            'unique_id' => $unique_id,
            'context'   => $context,
        ]);
    }

    /**
     * Формирование задачи на забор баланса
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initFetchBalanceTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_FETCH_BALANCE, $context);
    }

    /**
     * Формирование задачи на забор платежей
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initFetchPaymentsTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_FETCH_PAYMENTS, $context);
    }

    /**
     * Формирование задачи на забор профиля
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initFetchProfileTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_FETCH_PROFILE, $context);
    }

    /**
     * Формирование задачи на регистрацию
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initSignUpTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_SIGN_UP, $context);
    }

    /**
     * Формирование задачи на авторизацию
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initSignInTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_SIGN_IN, $context);
    }

    /**
     * Формирование задачи на подготовку
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initPreparationTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_PREPARATION, $context);
    }

    /**
     * Формирование задачи на создание платежа
     * @param array $context
     * @return \Astreya\TaskSystem\Models\QueuedTask
     */
    public function initCreatePaymentTask(array $context): QueuedTask
    {
        return $this->initQueuedTask(QueuedTaskData::TYPE_CREATE_PAYMENT, $context);
    }

    /**
     * Поиск существующей задачи по ключу
     * @param string $key
     * @return \Astreya\TaskSystem\Models\QueuedTask|null
     */
    public function findExistTask(string $key): ?QueuedTask
    {
        $id = Cache::get($key);

        if ($id === null)
            return null;

        return QueuedTask::find($id);
    }

    public function getTaskForProcess(string $type) : Collection
    {
        return QueuedTask::whereIn('state', [QueuedTaskData::STATE_IN_QUEUE, QueuedTaskData::STATE_IN_PROGRESS])
            ->where('type', '=', $type)->get();
    }
}